/******************************************************************************
efx_NSContrast.cpp

Copyright John Paul Chacha. All Rights Reserved.

This source code file, which has been provided by John Paul Chacha as part of 
his software product for use only by licensed users of that product, includes 
proprietary information of John Paul Chacha.

USE OF THIS SOFTWARE IS GOVERNED BY THE TERMS AND CONDITIONS OF THE LICENSE 
AGREEMENT FURNISHED WITH THE PRODUCT.

IN PARTICULAR, JOHN PAUL CHACHA SHALL BE FREE FROM ANY CLAIMS OR LIABILITIES 
ARISING OUT OF THE USE OR MISUSE OF THIS FILE AND/OR ITS CONTENTS.
******************************************************************************/
#include "../common/system.h"
#include "../common/plugin.h"
#include "../common/suite_info.h"
#include "../common/win_util.h"
#include "../common/str_util.h"
#include "resource.h"


//=============================================================================
#define PLUGIN_NAME		L"Non-saturating Contrast"
#define PLUGIN_AUTHOR	SUITE_CREATOR
#define PLUGIN_VERSION	SUITE_VER_INT32

#define MY_STORE_NAME	L"efx_NSContrast.dll"
#define MY_STORE_UUID	(*((LPDWORD)("_NSC")))

#define APP_MSG_NOTIFY			WM_APP+ 1

//-----------------------------------------------------------------------------
struct tmp_settings_tag
{
	unsigned long uuid;
	int power;
};

//-----------------------------------------------------------------------------
HINSTANCE	api_inst;
efx_IMAGE_T*		lp_host=0;
tmp_settings_tag	tmp_settings={0,};

LRESULT CALLBACK msg_dlg_about (HWND hDlg,UINT msg,WPARAM wpar,LPARAM lpar);
LRESULT CALLBACK msg_dlg_effect(HWND hDlg,UINT msg,WPARAM wpar,LPARAM lpar);

//=============================================================================
//=============================================================================



/******************************************************************************
This function is called during dll startup/shutdown.
******************************************************************************/
BOOL APIENTRY DllMain(
	HANDLE hModule,unsigned long ul_reason_for_call,LPVOID lpReserved)
{
	switch(ul_reason_for_call)
	{
	case DLL_PROCESS_ATTACH:
		api_inst=(HINSTANCE)hModule;
		break;
	case DLL_THREAD_ATTACH:
		break;
	case DLL_THREAD_DETACH:
		break;
	case DLL_PROCESS_DETACH:
		break;
	}
	UNREFERENCED_PARAMETER(lpReserved);
	return TRUE;
}
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////


/******************************************************************************
This function is called to query plugin's name and type.
******************************************************************************/
int plg_GetInfo(plg_INFO* info)
{
	info->api_type=PLUGIN_APITYPE_EFFECT;
	info->api_version=PLUGIN_INTERFACE_VERSION;
	info->plugin_version=PLUGIN_VERSION;
	wcsncpy(info->plugin_name,PLUGIN_NAME,63);
	info->plugin_name[63]=0;
	wcsncpy(info->plugin_author,PLUGIN_AUTHOR,63);
	info->plugin_author[63]=0;
	return 1;
}
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////


/******************************************************************************
This function is called to display plugin dialog boxes
******************************************************************************/
int plg_ShowDialog(plg_DIALOG* data)
{
	if(!data)
	{
		return PLUGIN_ERR_BAD_PARAM;
	}
	switch(data->dialog)
	{
	case PLG_DIALOG_ABOUT:
		DialogBoxParam(api_inst,(LPCTSTR)IDD_ABOUT,data->hwnd,
			(DLGPROC)msg_dlg_about,0);
		return 1;
	case PLG_DIALOG_CONFIG:
		return PLUGIN_ERR_NO_SUPPORT;
	case PLG_DIALOG_HELP:
		return PLUGIN_ERR_NO_SUPPORT;
	}
	return PLUGIN_ERR_BAD_PARAM;
}
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////


/******************************************************************************
This is the plugin DLL function that Chasys Draw IES calls to run the effect 
exposed by this plugin
******************************************************************************/
int efx_DoEffect(efx_IMAGE_T* data)
{
	int i,ret;

	//-------------------------------------------------------------------------
	if(!data)
	{
		return PLUGIN_ERR_BAD_HOST;
	}
	if(!data->pi_BasicImage)
	{
		return PLUGIN_ERR_BAD_HOST;
	}
	if(data->pi_BasicImage->version<PI_BASICIMAGE_VERSION)
	{
		return PLUGIN_ERR_BAD_HOST;
	}

	//-------------------------------------------------------------------------
	tmp_settings.power=100;
	if(data->pi_StateStore)
	{
		i=data->pi_StateStore->temp_size(MY_STORE_NAME);
		if(i>=sizeof(tmp_settings_tag))
		{
			data->pi_StateStore->temp_read(
				MY_STORE_NAME,&tmp_settings,sizeof(tmp_settings_tag));
		}
	}

	//-------------------------------------------------------------------------
	lp_host=data;
	lp_host->load();
	ret=DialogBoxParam(api_inst,(LPCTSTR)IDD_EFFECT,data->hwnd,
		(DLGPROC)msg_dlg_effect,0);
	if(ret==0)
	{
		return PLUGIN_ERR_USER_CANCEL;
	}
	if(ret<0)
	{
		return PLUGIN_ERR_INTERNAL;
	}

	//-------------------------------------------------------------------------
	if(data->pi_StateStore)
	{
		data->pi_StateStore->temp_write(
			MY_STORE_NAME,&tmp_settings,sizeof(tmp_settings_tag));
	}
	return PLUGIN_OKAY;
}
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////


/******************************************************************************
******************************************************************************/
int effect_Execute(int power)
{
	int i;
	unsigned char xlat[256];
	pibi_IMAGE my_surf;

	//-------------------------------------------------------------------------
	//reload original every time we update
	lp_host->load();

	//-------------------------------------------------------------------------
	//calc xlat tables
	for(i=0;i<=128;i++)
	{
		xlat[i]=(BYTE)(128.0*pow((i/128.0),1.0+(power/100.0)));
	}
	for(i=129;i<256;i++)
	{
		xlat[i]=(BYTE)(255-xlat[255-i]);
	}

	//-------------------------------------------------------------------------
	//lock bits for access
	lp_host->lock();

	lp_host->pi_BasicImage->virtualize(&my_surf,
		lp_host->width,lp_host->height,
		lp_host->pitch,lp_host->lp_pix);

	//-------------------------------------------------------------------------
	//execute
	lp_host->pi_BasicImage->translate(
		&my_surf,0,0,my_surf.width-1,my_surf.height-1,xlat,xlat,xlat,0,0,0);

	//-------------------------------------------------------------------------
	//unlock bits after access
	lp_host->unlock();

	//-------------------------------------------------------------------------
	//force host to refresh its display to reflect changes
	lp_host->refresh();
	//-------------------------------------------------------------------------
	return 1;
}
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////


/******************************************************************************
******************************************************************************/
LRESULT CALLBACK msg_dlg_effect(HWND hDlg,UINT msg,WPARAM wpar,LPARAM lpar)
{
	SCROLLINFO si;
	//-------------------------------------------------------------------------
	switch(msg)
	{
	//-------------------------------------------------------------------------
	case WM_INITDIALOG:
		SetWindowText(hDlg,PLUGIN_NAME);
		SetWindowLong(hDlg,GWL_EXSTYLE,WS_EX_TOPMOST|
			GetWindowLong(hDlg,GWL_EXSTYLE));
		memset(&si,0,sizeof(si));
		si.cbSize=sizeof(si);
		si.fMask=SIF_PAGE|SIF_POS|SIF_RANGE;
		si.nPage=100;
		si.nMin=0;
		si.nMax=1000;
		si.nPos=tmp_settings.power;
		SetScrollInfo(GetDlgItem(hDlg,IDC_SCROLL1),SB_CTL,&si,TRUE);
		lp_host->dock(hDlg);
		//---------------------------------------------------------------------
		effect_Execute(tmp_settings.power);
		return 1|lpar;
	//-------------------------------------------------------------------------
	case WM_HSCROLL:
		memset(&si,0,sizeof(si));
		si.cbSize=sizeof(si);
		si.fMask=SIF_PAGE|SIF_POS|SIF_RANGE|SIF_TRACKPOS;
		GetScrollInfo(GetDlgItem(hDlg,IDC_SCROLL1),SB_CTL,&si);
		si.nPage=100;
		si.nMin=0;
		si.nMax=1000;
		//---------------------------------------------------------------------
		switch(LOWORD(wpar))
		{
		case SB_LEFT:
			si.nPos=si.nMin;
            break;
		case SB_RIGHT:
			si.nPos=si.nMax;
			break;
		case SB_LINELEFT:
			si.nPos-=1;
			break;
		case SB_LINERIGHT:
			si.nPos+=1;
            break;
		case SB_PAGELEFT:
			si.nPos-=100;
            break;
		case SB_PAGERIGHT:
			si.nPos+=100;
			break;
		case SB_THUMBTRACK:
			si.nPos=si.nTrackPos;
			break;
		case SB_ENDSCROLL:
			break;
		}
		//---------------------------------------------------------------------
		if(si.nPos<si.nMin) si.nPos=si.nMin;
		si.fMask=SIF_PAGE|SIF_POS|SIF_RANGE;
		SetScrollInfo(GetDlgItem(hDlg,IDC_SCROLL1),SB_CTL,&si,TRUE);
		GetScrollInfo(GetDlgItem(hDlg,IDC_SCROLL1),SB_CTL,&si);
		tmp_settings.power=si.nPos;
		//---------------------------------------------------------------------
		effect_Execute(tmp_settings.power);
		break;
	//-------------------------------------------------------------------------
	case WM_COMMAND:
		switch(LOWORD(wpar))
		{
		//---------------------------------------------------------------------
		case IDOK    :
			//-----------------------------------------------------------------
			effect_Execute(tmp_settings.power);
			EndDialog(hDlg,1);
			break;
		//---------------------------------------------------------------------
		case IDCANCEL:
			EndDialog(hDlg,0);
			break;
		}
		return 1;
	}
	//-------------------------------------------------------------------------
	return 0;
}
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////


/******************************************************************************
This function processes messages for the about dialog
******************************************************************************/
LRESULT CALLBACK msg_dlg_about(HWND hDlg,UINT msg,WPARAM wpar,LPARAM lpar)
{
	wchar_t* pstr;

	//-------------------------------------------------------------------------
	switch(msg)
	{
	case WM_INITDIALOG:
		SetWindowText(hDlg,L"About " PLUGIN_NAME);
		pstr=string_ReadTextResource(api_inst,(LPCTSTR)IDR_README,L"TEXT");
		if(pstr)
		{
			SetDlgItemText(hDlg,IDC_EDIT1,pstr);
			GlobalFree(pstr);
		}
		window_Center(hDlg,GetParent(hDlg));
		return 1|lpar;
	case WM_COMMAND:
		switch(LOWORD(wpar))
		{
		case IDOK    :
			EndDialog(hDlg,1);
			break;
		case IDCANCEL:
			EndDialog(hDlg,0);
			break;
		}
		return 1;
	}
	return 0;
}
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////