#include <QDateTime>
#include <QStringList>
#include <QRegExp>
#include <QSqlQuery>
#include <QDomDocument>
#include <QNetworkRequest>
#include <QNetworkCookie>
#include <QMessageBox>
#include <QWebHistory>
#include <QDesktopServices>
#include <QCoreApplication>

#include <qsysteminfo.h>

#include "appcontroller.h"
#include "downloadslistitemwidget.h"
#include "ui_appcontroller.h"

QTM_USE_NAMESPACE;

YTNetworkCookieJar::YTNetworkCookieJar(QObject *parent) : QNetworkCookieJar(parent)
{
    QList<QNetworkCookie> cookie_list;

    CookieDatabase = QSqlDatabase::addDatabase("QSQLITE");

    CookieDatabase.setDatabaseName("cookie.db");

    if (CookieDatabase.open()) {
        QSqlQuery query(CookieDatabase);

        if (query.exec("SELECT NAME, VALUE, DOMAIN, EXPIRATION_DATE, HTTP_ONLY, PATH, SECURE FROM COOKIES")) {
            while (query.next()) {
                QNetworkCookie cookie(query.value(0).toByteArray(), query.value(1).toByteArray());

                cookie.setDomain(query.value(2).toString());
                cookie.setExpirationDate(QDateTime::fromMSecsSinceEpoch(query.value(3).toLongLong()).toUTC());
                cookie.setHttpOnly(query.value(4).toBool());
                cookie.setPath(query.value(5).toString());
                cookie.setSecure(query.value(6).toBool());

                cookie_list.append(cookie);
            }
        }

        CookieDatabase.close();
    }

    setAllCookies(cookie_list);
}

YTNetworkCookieJar::~YTNetworkCookieJar()
{
    QList<QNetworkCookie> cookie_list = allCookies();

    CookieDatabase.setDatabaseName("cookie.db");

    if (CookieDatabase.open()) {
        QSqlQuery query(CookieDatabase);

        if (query.exec("CREATE TABLE IF NOT EXISTS COOKIES(NAME TEXT, VALUE TEXT, DOMAIN TEXT, EXPIRATION_DATE NUMERIC, HTTP_ONLY NUMERIC, PATH TEXT, SECURE NUMERIC)") &&
            query.exec("DELETE FROM COOKIES")) {
            query.prepare("INSERT INTO COOKIES(NAME, VALUE, DOMAIN, EXPIRATION_DATE, HTTP_ONLY, PATH, SECURE) VALUES (:NAME, :VALUE, :DOMAIN, :EXPIRATION_DATE, :HTTP_ONLY, :PATH, :SECURE)");

            for (int i = 0; i < cookie_list.size(); i++) {
                QNetworkCookie cookie = cookie_list.at(i);

                if (!cookie.isSessionCookie()) {
                    query.bindValue(":NAME", cookie.name());
                    query.bindValue(":VALUE", cookie.value());
                    query.bindValue(":DOMAIN", cookie.domain());
                    query.bindValue(":EXPIRATION_DATE", cookie.expirationDate().toUTC().toMSecsSinceEpoch());
                    query.bindValue(":HTTP_ONLY", cookie.isHttpOnly());
                    query.bindValue(":PATH", cookie.path());
                    query.bindValue(":SECURE", cookie.isSecure());

                    query.exec();
                }
            }
        }

        CookieDatabase.close();
    }
}

AppController::AppController(QWidget *parent) : QWidget(parent), ui(new Ui::AppController)
{
    VideoFormat format_spec;

    ui->setupUi(this);

    NetworkInitialization = true;
    WebViewInitialized    = false;
    WebViewLoading        = false;
    MetadataLoading       = false;
    PreviewLoading        = false;
    VideoId               = "";
    VideoTitle            = "";

    QString base64_stylesheet(QByteArray("body { -webkit-user-select: none; }").toBase64());

    ui->WebView->settings()->setUserStyleSheetUrl(QUrl(QString("data:text/css;charset=utf-8;base64,%1").arg(base64_stylesheet)));
    ui->WebView->settings()->setAttribute(QWebSettings::PluginsEnabled, true);

    format_spec.Fmt       = 22;
    format_spec.Name      = "MP4 (H.264 720p HD)";
    format_spec.Extension = "mp4";

    AllVideoFormats.append(format_spec);

    format_spec.Fmt       = 18;
    format_spec.Name      = "MP4 (H.264 360p)";
    format_spec.Extension = "mp4";

    AllVideoFormats.append(format_spec);

    format_spec.Fmt       = 35;
    format_spec.Name      = "FLV (H.264 480p)";
    format_spec.Extension = "flv";

    AllVideoFormats.append(format_spec);

    format_spec.Fmt       = 34;
    format_spec.Name      = "FLV (H.264 360p)";
    format_spec.Extension = "flv";

    AllVideoFormats.append(format_spec);

    format_spec.Fmt       = 5;
    format_spec.Name      = "FLV (H.263 240p)";
    format_spec.Extension = "flv";

    AllVideoFormats.append(format_spec);

    NetworkConfigurationManager = new QNetworkConfigurationManager();
    NetworkSession              = NULL;
    NetworkAccessManager        = ui->WebView->page()->networkAccessManager();
    AppSettings                 = new QSettings(QDir::current().path() + QDir::separator() + "config.ini", QSettings::IniFormat);
    DownloadMgr                 = new DownloadManager(NetworkAccessManager);
    MetadataReply               = NULL;
    PreviewReply                = NULL;

    connect(NetworkAccessManager,      SIGNAL(finished(QNetworkReply*)), SLOT(NetworkRequestFinished(QNetworkReply*)));
    connect(&DownloadsListUpdateTimer, SIGNAL(timeout()),                SLOT(UpdateDownloadsList()));

    NetworkAccessManager->setCookieJar(new YTNetworkCookieJar());

    DownloadsListUpdateTimer.setInterval(UPDATE_INTERVAL);
    DownloadsListUpdateTimer.start();

    ui->AppStackedWidget->setCurrentIndex(BROWSE_PAGE);

    QTimer::singleShot(0, this, SLOT(InitNetwork()));

    AdjustWidgets();
}

AppController::~AppController()
{
    if (MetadataReply != NULL) {
        MetadataReply->abort();
    }
    if (PreviewReply != NULL) {
        PreviewReply->abort();
    }

    for (int i = 0; i < ui->DownloadsListWidget->count(); i++) {
        QListWidgetItem *item   = ui->DownloadsListWidget->item(i);
        QWidget         *widget = ui->DownloadsListWidget->itemWidget(item);

        ui->DownloadsListWidget->removeItemWidget(item);

        delete widget;
    }

    delete DownloadMgr;
    delete AppSettings;

    if (NetworkSession != NULL) {
        delete NetworkSession;
    }

    delete NetworkConfigurationManager;

    delete ui;
}

void AppController::InitNetwork()
{
    QCoreApplication::processEvents();

    NetworkStateChanged(false);
}

void AppController::NetworkSessionOpened()
{
    NetworkStateChanged(true);
}

void AppController::NetworkSessionClosed()
{
    if (!NetworkInitialization) {
        NetworkStateChanged(false);
    }
}

void AppController::NetworkSessionError(QNetworkSession::SessionError)
{
    if (NetworkInitialization) {
        NetworkStateChanged(false);
    }
}

void AppController::UpdateDownloadsList()
{
    DownloadTasks = DownloadMgr->GetTasks();

    if (ui->DownloadsListWidget->currentRow() >= DownloadTasks.size()) {
        ui->DownloadsListWidget->setCurrentRow(DownloadTasks.size() - 1);
    }

    if (ui->DownloadsListWidget->count() < DownloadTasks.size()) {
        int count = DownloadTasks.size() - ui->DownloadsListWidget->count();

        for (int i = 0; i < count; i++) {
            QListWidgetItem         *item   = new QListWidgetItem();
            DownloadsListItemWidget *widget = new DownloadsListItemWidget();

            ui->DownloadsListWidget->addItem(item);
            ui->DownloadsListWidget->setItemWidget(item, widget);

            item->setSizeHint(widget->sizeHint());
        }
    } else if (ui->DownloadsListWidget->count() > DownloadTasks.size()) {
        int count = ui->DownloadsListWidget->count() - DownloadTasks.size();

        for (int i = 0; i < count; i++) {
            QListWidgetItem *item   = ui->DownloadsListWidget->item(ui->DownloadsListWidget->count() - 1);
            QWidget         *widget = ui->DownloadsListWidget->itemWidget(item);

            ui->DownloadsListWidget->removeItemWidget(item);

            item = ui->DownloadsListWidget->takeItem(ui->DownloadsListWidget->count() - 1);

            delete widget;
            delete item;
        }
    }

    for (int i = 0; i < DownloadTasks.size(); i++) {
        DownloadManager::DownloadTask task    = DownloadTasks.at(i);
        DownloadsListItemWidget       *widget = static_cast<DownloadsListItemWidget*>(ui->DownloadsListWidget->itemWidget(ui->DownloadsListWidget->item(i)));

        widget->GetTitleLabel()->setText(task.Title);

        if (task.State == DownloadManager::STATE_ACTIVE) {
            widget->GetStateLabel()->setVisible(false);
            widget->GetProgressBar()->setVisible(true);

            if (task.Size != 0) {
                widget->GetProgressBar()->setValue((task.Done * 100) / task.Size);
            } else {
                widget->GetProgressBar()->setValue(0);
            }
        } else {
            widget->GetStateLabel()->setVisible(true);
            widget->GetProgressBar()->setVisible(false);

            if (task.State == DownloadManager::STATE_COMPLETED) {
                widget->GetStateLabel()->setText("COMPLETED");
            } else if (task.State == DownloadManager::STATE_ERROR) {
                widget->GetStateLabel()->setText(task.ErrorMsg);
            } else if (task.State == DownloadManager::STATE_QUEUED) {
                widget->GetStateLabel()->setText("QUEUED");
            } else if (task.State == DownloadManager::STATE_PAUSED) {
                widget->GetStateLabel()->setText("PAUSED");
            }
        }
    }

    ui->DownloadsListWidget->update();

    AdjustWidgets();
}

void AppController::NetworkRequestFinished(QNetworkReply *reply)
{
    if (reply == MetadataReply) {
        MetadataLoading = false;

        if (reply->error() == QNetworkReply::NoError) {
            QHash<int, QString> fmt_url_map;

            if (DownloadMgr->ParseMetadata(reply->readAll(), &VideoTitle, &fmt_url_map)) {
                AvailableVideoFormats.clear();

                for (int i = 0; i < AllVideoFormats.size(); i++) {
                    if (fmt_url_map.contains(AllVideoFormats.at(i).Fmt)) {
                        VideoFormat video_format;

                        video_format.Fmt       = AllVideoFormats.at(i).Fmt;
                        video_format.Name      = AllVideoFormats.at(i).Name;
                        video_format.Extension = AllVideoFormats.at(i).Extension;

                        AvailableVideoFormats.append(video_format);
                    }
                }
                if (AvailableVideoFormats.size() > 0) {
                    AvailableDisks.clear();

                    QSystemStorageInfo sys_info;
                    QStringList        disk_names = QSystemStorageInfo::logicalDrives();
                    QList<QIcon>       disk_icons;

                    for (int i = 0; i < disk_names.size(); i++) {
                        if (sys_info.typeForDrive(disk_names.at(i)) == QSystemStorageInfo::InternalDrive) {
                            AvailableDisks.append(disk_names.at(i));

                            disk_icons.append(QIcon(":/resources/images/drive_internal.png"));
                        } else if (sys_info.typeForDrive(disk_names.at(i)) == QSystemStorageInfo::RemovableDrive) {
                            AvailableDisks.append(disk_names.at(i));

                            disk_icons.append(QIcon(":/resources/images/drive_removable.png"));
                        } else if (sys_info.typeForDrive(disk_names.at(i)) == QSystemStorageInfo::RemoteDrive) {
                            AvailableDisks.append(disk_names.at(i));

                            disk_icons.append(QIcon(":/resources/images/drive_remote.png"));
                        }
                    }

                    if (AvailableDisks.size() > 0) {
                        QString file_name;

                        for (int i = 0; i < MAX_FNAME_DUPCHECK_TRIES; i++) {
                            if (i == 0) {
                                file_name = MakeValidFilename(VideoTitle);
                            } else {
                                file_name = MakeValidFilename(QString("%1 (%2)").arg(VideoTitle).arg(i));
                            }

                            if (!DownloadMgr->HaveDuplicate(file_name)) {
                                break;
                            }
                        }

                        ui->FileNameLineEdit->setText(file_name);

                        ui->DestinationDiskComboBox->clear();

                        int      selected_dst_disk = 0;
                        QVariant settings_dst_disk = AppSettings->value("DestinationDisk");

                        for (int i = 0; i < AvailableDisks.size(); i++) {
                            ui->DestinationDiskComboBox->addItem(disk_icons.at(i), AvailableDisks.at(i) + " [" +
                                                                 MakeHumanReadableSize(sys_info.availableDiskSpace(AvailableDisks.at(i))) + "/" +
                                                                 MakeHumanReadableSize(sys_info.totalDiskSpace(AvailableDisks.at(i))) + "]");

                            if (!settings_dst_disk.isNull() && settings_dst_disk.toString() == AvailableDisks.at(i)) {
                                selected_dst_disk = i;
                            }
                        }

                        ui->DestinationDiskComboBox->setCurrentIndex(selected_dst_disk);

                        for (int i = 0; i < VideoFormatRadioButtons.size(); i++) {
                            ui->VideoFormatGroupBoxLayout->removeWidget(VideoFormatRadioButtons.at(i));

                            delete VideoFormatRadioButtons.at(i);
                        }

                        VideoFormatRadioButtons.clear();

                        int      selected_format = 0;
                        QVariant settings_format = AppSettings->value("VideoFormat");

                        for (int i = 0; i < AvailableVideoFormats.size(); i++) {
                            QRadioButton *button = new QRadioButton(AvailableVideoFormats.at(i).Name);

                            if (!settings_format.isNull() && settings_format.toInt() == AvailableVideoFormats.at(i).Fmt) {
                                selected_format = i;
                            }

                            VideoFormatRadioButtons.append(button);

                            ui->VideoFormatGroupBoxLayout->addWidget(button);
                        }

                        VideoFormatRadioButtons.at(selected_format)->setChecked(true);

                        ui->AppStackedWidget->setCurrentIndex(DOWNLOAD_PAGE);
                    } else {
                        QMessageBox::critical(this, "Error", "No suitable disks found");
                    }
                } else {
                    QMessageBox::critical(this, "Error", "No video formats are available for download");
                }
            } else {
                QMessageBox::critical(this, "Error", "No video formats are available for download");
            }
        } else if (reply->error() != QNetworkReply::OperationCanceledError) {
            QMessageBox::critical(this, "Error", "No video formats are available for download");
        }

        MetadataReply = NULL;

        reply->deleteLater();

        AdjustWidgets();
    } else if (reply == PreviewReply) {
        PreviewLoading = false;

        if (reply->error() == QNetworkReply::NoError) {
            QDomDocument document;

            if (document.setContent(reply->readAll())) {
                QString      preview_url  = "";
                QDomElement  root         = document.documentElement();
                QDomNodeList media_groups = root.elementsByTagName("media:group");

                for (int i = 0; i < media_groups.size(); i++) {
                    QDomNode entry = media_groups.at(i).toElement().firstChild();

                    while (!entry.isNull()) {
                        if (entry.toElement().tagName() == "media:content") {
                            if (entry.toElement().attribute("type")   == "video/3gpp" &&
                                entry.toElement().attribute("medium") == "video") {
                                preview_url = entry.toElement().attribute("url");
                            }
                        }

                        if (preview_url.isEmpty()) {
                            entry = entry.nextSibling();
                        } else {
                            break;
                        }
                    }

                    if (!preview_url.isEmpty()) {
                        break;
                    }
                }

                if (!preview_url.isEmpty()) {
                    if (!QDesktopServices::openUrl(QUrl::fromEncoded(preview_url.toAscii()))) {
                        QMessageBox::critical(this, "Error", "Failed to start video player");
                    }
                } else {
                    QMessageBox::critical(this, "Error", "Error loading video metadata");
                }
            } else {
                QMessageBox::critical(this, "Error", "Error loading video metadata");
            }
        } else if (reply->error() != QNetworkReply::OperationCanceledError) {
            QMessageBox::critical(this, "Error", "Error loading video metadata");
        }

        PreviewReply = NULL;

        reply->deleteLater();

        AdjustWidgets();
    }
}

void AppController::on_WebView_loadStarted()
{
    WebViewLoading = true;

    AdjustWidgets();
}

void AppController::on_WebView_loadFinished(bool)
{
    WebViewLoading = false;

    QString url         = ui->WebView->url().toString();
    QRegExp validator   = QRegExp("/watch\\?");
    QRegExp extractor_1 = QRegExp("&v=([^&]+)");
    QRegExp extractor_2 = QRegExp("\\?v=([^&]+)");

    if (validator.indexIn(url) != -1) {
        if (extractor_1.indexIn(url) != -1) {
            VideoId = extractor_1.cap(1);
        } else if (extractor_1.indexIn(url) != -1) {
            VideoId = extractor_2.cap(1);
        } else {
            VideoId = "";
        }
    } else {
        VideoId = "";
    }

    AdjustWidgets();
}

void AppController::on_NavigateButton_toggled(bool)
{
    AdjustWidgets();
}

void AppController::on_DownloadButton_clicked()
{
    if (VideoId != "") {
        if (MetadataReply == NULL) {
            MetadataLoading = true;

            QUrl url = QUrl::fromEncoded(QString("http://www.youtube.com/watch?v=%1&nomobile=1").arg(VideoId).toAscii());

            MetadataReply = NetworkAccessManager->get(QNetworkRequest(url));
        }
    }

    AdjustWidgets();
}

void AppController::on_DownloadsButton_clicked()
{
    ui->AppStackedWidget->setCurrentIndex(DOWNLOADS_PAGE);

    AdjustWidgets();
}

void AppController::on_BrowsePageExitButton_clicked()
{
    if (QMessageBox::question(this, "Quit", "Are you sure you want to quit?", QMessageBox::Yes | QMessageBox::No) == QMessageBox::Yes) {
        QCoreApplication::quit();
    }
}

void AppController::on_BackStopButton_clicked()
{
    if (WebViewLoading) {
        ui->WebView->stop();
    } else {
        if (MetadataLoading) {
            MetadataLoading = false;

            if (MetadataReply != NULL) {
                MetadataReply->abort();
            }

            AdjustWidgets();
        } else if (PreviewLoading) {
            PreviewLoading = false;

            if (PreviewReply != NULL) {
                PreviewReply->abort();
            }

            AdjustWidgets();
        } else {
            if (ui->WebView->history()->canGoBack() && ui->WebView->history()->backItem().url() != QUrl("about:blank")) {
                ui->WebView->back();
            }
        }
    }
}

void AppController::on_ReloadButton_clicked()
{
    ui->WebView->reload();
}

void AppController::on_PreviewButton_clicked()
{
    if (VideoId != "") {
        if (PreviewReply == NULL) {
            PreviewLoading = true;

            QUrl url = QUrl::fromEncoded(QString("http://gdata.youtube.com/feeds/api/videos/%1").arg(VideoId).toAscii());

            PreviewReply = NetworkAccessManager->get(QNetworkRequest(url));
        }
    }

    AdjustWidgets();
}

void AppController::on_HelpButton_clicked()
{
    ui->WebView->load(QUrl::fromEncoded("qrc:/resources/doc/help.html"));
}

void AppController::on_DownloadOKButton_clicked()
{
    int     fmt             = 0;
    QString file_name       = "",
            file_extension  = "",
            dst_disk        = "",
            validate_result = "";

    file_name = ui->FileNameLineEdit->text();

    if (ui->DestinationDiskComboBox->currentIndex() != -1) {
        dst_disk = AvailableDisks.at(ui->DestinationDiskComboBox->currentIndex());

        AppSettings->setValue("DestinationDisk", dst_disk);
    }

    for (int i = 0; i < VideoFormatRadioButtons.size(); i++) {
        if (VideoFormatRadioButtons.at(i)->isChecked()) {
            fmt            = AvailableVideoFormats.at(i).Fmt;
            file_extension = AvailableVideoFormats.at(i).Extension;

            AppSettings->setValue("VideoFormat", AvailableVideoFormats.at(i).Fmt);
        }
    }

    if (!file_name.isEmpty()) {
        if (!file_extension.isEmpty()) {
            if (!dst_disk.isEmpty()) {
                if (DownloadMgr->AddTask(file_name, VideoId, fmt, file_extension, dst_disk, VideoTitle, &validate_result)) {
                    ui->AppStackedWidget->setCurrentIndex(DOWNLOADS_PAGE);

                    AdjustWidgets();
                } else {
                    QMessageBox::critical(this, "Error", validate_result);
                }
            } else {
                QMessageBox::critical(this, "Error", "No destination disk selected");
            }
        } else {
            QMessageBox::critical(this, "Error", "No video format selected");
        }
    } else {
        QMessageBox::critical(this, "Error", "File name is empty");
    }
}

void AppController::on_DownloadCancelButton_clicked()
{
    ui->AppStackedWidget->setCurrentIndex(BROWSE_PAGE);

    AdjustWidgets();
}

void AppController::on_DeleteDownloadButton_clicked()
{
    if (ui->DownloadsListWidget->currentRow() != -1) {
        if (QMessageBox::question(this, "Delete download", "Are you sure you want to delete?", QMessageBox::Yes | QMessageBox::No) == QMessageBox::Yes) {
            QString file_name = DownloadTasks.at(ui->DownloadsListWidget->currentRow()).FileName;

            DownloadMgr->DelTask(file_name);
        }
    }
}

void AppController::on_PlayPauseButton_clicked()
{
    if (ui->DownloadsListWidget->currentRow() != -1) {
        DownloadManager::DownloadTask task = DownloadTasks.at(ui->DownloadsListWidget->currentRow());

        if (task.State == DownloadManager::STATE_COMPLETED) {
            if (!QDesktopServices::openUrl(QUrl::fromLocalFile(DownloadMgr->MakeFullFilenameForTask(task)))) {
                QMessageBox::critical(this, "Error", "Failed to start video player");
            }
        } else if (task.State == DownloadManager::STATE_PAUSED) {
            DownloadMgr->ResumeTask(task.FileName);
        } else {
            DownloadMgr->PauseTask(task.FileName);
        }
    }
}

void AppController::on_BrowseButton_clicked()
{
    ui->AppStackedWidget->setCurrentIndex(BROWSE_PAGE);

    AdjustWidgets();
}

void AppController::on_DownloadsPageExitButton_clicked()
{
    if (QMessageBox::question(this, "Quit", "Are you sure you want to quit?", QMessageBox::Yes | QMessageBox::No) == QMessageBox::Yes) {
        QCoreApplication::quit();
    }
}

void AppController::on_DownloadsListWidget_currentRowChanged(int)
{
    AdjustWidgets();
}

void AppController::on_DownloadsListWidget_itemDoubleClicked(QListWidgetItem* item)
{
    DownloadManager::DownloadTask task = DownloadTasks.at(ui->DownloadsListWidget->row(item));

    if (task.State == DownloadManager::STATE_COMPLETED) {
        if (!QDesktopServices::openUrl(QUrl::fromLocalFile(DownloadMgr->MakeFullFilenameForTask(task)))) {
            QMessageBox::critical(this, "Error", "Failed to start video player");
        }
    }
}

void AppController::NetworkStateChanged(bool is_online)
{
    if (is_online) {
        NetworkInitialization = false;

        DownloadMgr->SetNetworkAvailability(true);

        NetworkAccessManager->setConfiguration(NetworkConfigurationManager->configurationFromIdentifier(NetworkSession->sessionProperty("ActiveConfiguration").toString()));

        if (!WebViewInitialized) {
            ui->WebView->load(QUrl::fromEncoded("http://m.youtube.com/?mobileflash=0"));

            WebViewInitialized = true;
        }
    } else {
        NetworkInitialization = true;

        DownloadMgr->SetNetworkAvailability(false);

        NetworkAccessManager->setConfiguration(QNetworkConfiguration());

        if (NetworkSession != NULL) {
            NetworkSession->deleteLater();
        }

        NetworkSession = new QNetworkSession(NetworkConfigurationManager->defaultConfiguration());

        connect(NetworkSession, SIGNAL(opened()),                             SLOT(NetworkSessionOpened()));
        connect(NetworkSession, SIGNAL(closed()),                             SLOT(NetworkSessionClosed()));
        connect(NetworkSession, SIGNAL(error(QNetworkSession::SessionError)), SLOT(NetworkSessionError(QNetworkSession::SessionError)));

        NetworkSession->open();
    }

    AdjustWidgets();
}

void AppController::AdjustWidgets()
{
    if (NetworkInitialization || MetadataLoading || PreviewLoading) {
        ui->WebView->setEnabled(false);
    } else {
        ui->WebView->setEnabled(true);
    }
    if (NetworkInitialization || WebViewLoading || MetadataLoading || PreviewLoading) {
        ui->ProgressBar->setVisible(true);
    } else {
        ui->ProgressBar->setVisible(false);
    }
    if (ui->NavigateButton->isChecked()) {
        ui->BackStopButton->setVisible(true);
        ui->ReloadButton->setVisible(true);
        ui->PreviewButton->setVisible(true);
        ui->HelpButton->setVisible(true);
    } else {
        ui->BackStopButton->setVisible(false);
        ui->ReloadButton->setVisible(false);
        ui->PreviewButton->setVisible(false);
        ui->HelpButton->setVisible(false);
    }
    if (NetworkInitialization || WebViewLoading || MetadataLoading || PreviewLoading) {
        ui->DownloadButton->setEnabled(false);
    } else {
        if (VideoId != "") {
            ui->DownloadButton->setEnabled(true);
        } else {
            ui->DownloadButton->setEnabled(false);
        }
    }
    if (MetadataLoading || PreviewLoading) {
        ui->DownloadsButton->setEnabled(false);
    } else {
        ui->DownloadsButton->setEnabled(true);
    }
    if (NetworkInitialization) {
        ui->BackStopButton->setEnabled(false);
    } else if (WebViewLoading || MetadataLoading || PreviewLoading) {
        ui->BackStopButton->setIcon(QIcon(":/resources/images/stop.png"));

        ui->BackStopButton->setEnabled(true);
    } else {
        ui->BackStopButton->setIcon(QIcon(":/resources/images/back.png"));

        if (ui->WebView->history()->canGoBack() && ui->WebView->history()->backItem().url() != QUrl("about:blank")) {
            ui->BackStopButton->setEnabled(true);
        } else {
            ui->BackStopButton->setEnabled(false);
        }
    }
    if (NetworkInitialization || WebViewLoading || MetadataLoading || PreviewLoading) {
        ui->PreviewButton->setEnabled(false);
    } else {
        if (VideoId != "") {
            ui->PreviewButton->setEnabled(true);
        } else {
            ui->PreviewButton->setEnabled(false);
        }
    }
    if (NetworkInitialization || WebViewLoading || MetadataLoading || PreviewLoading) {
        ui->ReloadButton->setEnabled(false);
        ui->HelpButton->setEnabled(false);
    } else {
        ui->ReloadButton->setEnabled(true);
        ui->HelpButton->setEnabled(true);
    }
    if (ui->DownloadsListWidget->currentRow() != -1) {
        ui->DeleteDownloadButton->setEnabled(true);
    } else {
        ui->DeleteDownloadButton->setEnabled(false);
    }
    if (ui->DownloadsListWidget->currentRow() != -1) {
        if (DownloadTasks.at(ui->DownloadsListWidget->currentRow()).State == DownloadManager::STATE_COMPLETED) {
            ui->PlayPauseButton->setIcon(QIcon(":/resources/images/play_video.png"));
        } else {
            ui->PlayPauseButton->setIcon(QIcon(":/resources/images/play_pause.png"));
        }

        ui->PlayPauseButton->setEnabled(true);
    } else {
        ui->PlayPauseButton->setIcon(QIcon(":/resources/images/play_pause.png"));

        ui->PlayPauseButton->setEnabled(false);
    }
}

QString AppController::MakeValidFilename(const QString &string)
{
    QString result(string);

    result = result.replace("\\", "_");
    result = result.replace("/",  "_");
    result = result.replace(":",  "_");
    result = result.replace("*",  "_");
    result = result.replace("?",  "_");
    result = result.replace("\"", "_");
    result = result.replace("'",  "_");
    result = result.replace(">",  "_");
    result = result.replace("<",  "_");
    result = result.replace("|",  "_");

    return result;
}

QString AppController::MakeHumanReadableSize(qint64 bytes)
{
    QString result;

    if (bytes >= 1000000000) {
        result = QString("%1 GB").arg(bytes / 1000000000);
    } else if (bytes >= 1000000) {
        result = QString("%1 MB").arg(bytes / 1000000);
    } else if (bytes >= 1000) {
        result = QString("%1 KB").arg(bytes / 1000);
    } else {
        result = QString("%1 Bytes").arg(bytes);
    }

    return result;
}
