#include <QStringList>
#include <QRegExp>
#include <QNetworkRequest>
#include <QTimer>
#include <QSettings>

#include "downloadmanager.h"

DownloadManager::DownloadManager(QNetworkAccessManager *network_access_manager, QObject *parent) : QObject(parent)
{
    NetworkAvailable = false;

    DownloadsDir = QDir(QDir::current().path() + QDir::separator() + "downloads");

    if (DownloadsDir.exists()) {
        QStringList file_names = DownloadsDir.entryList(QDir::Files | QDir::Readable | QDir::Writable);

        for (int i = 0; i < file_names.size(); i++) {
            DownloadTask task;

            if (LoadTask(file_names.at(i), &task)) {
                DownloadTasks.append(task);
            }
        }
    } else {
        QDir::current().mkdir("downloads");
    }

    NetworkAccessManager = network_access_manager;
    MetadataReply        = NULL;
    DownloadReply        = NULL;

    connect(NetworkAccessManager, SIGNAL(finished(QNetworkReply*)), SLOT(NetworkRequestFinished(QNetworkReply*)));

    QTimer::singleShot(QUEUE_RUN_AFTER, this, SLOT(RunQueue()));
}

DownloadManager::~DownloadManager()
{
    if (MetadataReply != NULL) {
        MetadataReply->abort();
    }
    if (DownloadReply != NULL) {
        DownloadReply->abort();
    }

    for (int i = 0; i < DownloadTasks.size(); i++) {
        SaveTask(DownloadTasks.at(i));
    }
}

void DownloadManager::SetNetworkAvailability(bool network_available)
{
    NetworkAvailable = network_available;
}

const QList<DownloadManager::DownloadTask> &DownloadManager::GetTasks()
{
    return DownloadTasks;
}

bool DownloadManager::AddTask(const QString &file_name, const QString &video_id, int fmt, const QString &file_extension, const QString &destination_disk, const QString &title, QString *validate_error)
{
    if (file_name.isEmpty()) {
        *validate_error = "Download task file name is empty";

        return false;
    } else if (video_id.isEmpty()) {
        *validate_error = "Download task video id is empty";

        return false;
    } else if (file_extension.isEmpty()) {
        *validate_error = "Download task file extension is empty";

        return false;
    } else if (destination_disk.isEmpty()) {
        *validate_error = "Download task destination disk is empty";

        return false;
    } else if (title.isEmpty()) {
        *validate_error = "Download task title is empty";

        return false;
    } else {
        for (int i = 0; i < DownloadTasks.size(); i++) {
            if (DownloadTasks.at(i).FileName.compare(file_name, Qt::CaseInsensitive) == 0) {
                *validate_error = "Download task with the same file name already exists";

                return false;
            }
        }

        DownloadTask task;

        task.State           = STATE_QUEUED;
        task.Fmt             = fmt;
        task.Size            = 0;
        task.Done            = 0;
        task.FileName        = file_name;
        task.VideoId         = video_id;
        task.FileExtension   = file_extension;
        task.DestinationDisk = destination_disk;
        task.Title           = title;
        task.ErrorMsg        = "";

        DownloadTasks.append(task);

        QTimer::singleShot(QUEUE_RUN_AFTER, this, SLOT(RunQueue()));

        return true;
    }
}

void DownloadManager::DelTask(const QString &file_name)
{
    for (int i = 0; i < DownloadTasks.size(); i++) {
        if (DownloadTasks.at(i).FileName == file_name) {
            if (CurrentTask.FileName == file_name) {
                if (MetadataReply != NULL) {
                    MetadataReply->abort();
                }
                if (DownloadReply != NULL) {
                    DownloadReply->abort();
                }
            }

            ClearTask(DownloadTasks.at(i));

            DownloadTasks.removeAt(i);

            break;
        }
    }

    QTimer::singleShot(QUEUE_RUN_AFTER, this, SLOT(RunQueue()));
}

void DownloadManager::PauseTask(const QString &file_name)
{
    for (int i = 0; i < DownloadTasks.size(); i++) {
        if (DownloadTasks.at(i).FileName == file_name && DownloadTasks.at(i).State != STATE_COMPLETED &&
                                                         DownloadTasks.at(i).State != STATE_PAUSED) {
            if (CurrentTask.FileName == file_name) {
                if (MetadataReply != NULL) {
                    MetadataReply->abort();
                }
                if (DownloadReply != NULL) {
                    DownloadReply->abort();
                }
            }

            DownloadTask task = DownloadTasks.at(i);

            task.State = STATE_PAUSED;

            DownloadTasks.replace(i, task);

            break;
        }
    }

    QTimer::singleShot(QUEUE_RUN_AFTER, this, SLOT(RunQueue()));
}

void DownloadManager::ResumeTask(const QString &file_name)
{
    for (int i = 0; i < DownloadTasks.size(); i++) {
        if (DownloadTasks.at(i).FileName == file_name && DownloadTasks.at(i).State == STATE_PAUSED) {
            DownloadTask task = DownloadTasks.at(i);

            task.State = STATE_QUEUED;

            DownloadTasks.replace(i, task);

            break;
        }
    }

    QTimer::singleShot(QUEUE_RUN_AFTER, this, SLOT(RunQueue()));
}

bool DownloadManager::HaveDuplicate(const QString &file_name)
{
    for (int i = 0; i < DownloadTasks.size(); i++) {
        if (DownloadTasks.at(i).FileName.compare(file_name, Qt::CaseInsensitive) == 0) {
            return true;
        }
    }

    return false;
}

QString DownloadManager::MakeFullFilenameForTask(const DownloadTask &task)
{
    QString result   = "";
    QString dst_disk = task.DestinationDisk;

    if (!dst_disk.endsWith(QDir::separator())) {
        if (dst_disk.endsWith(":")) {
            dst_disk = dst_disk + QDir::separator();
        } else {
            dst_disk = dst_disk + ":" + QDir::separator();
        }
    }

    QDir root        = QDir(dst_disk);
    QDir data        = QDir(dst_disk + "DATA");
    QDir data_videos = QDir(dst_disk + "DATA" + QDir::separator() + "Videos");
    QDir videos      = QDir(dst_disk + "Videos");

    if (data_videos.exists()) {
        result = data_videos.path() + QDir::separator() + task.FileName + "." + task.FileExtension;
    } else if (videos.exists()) {
        result = videos.path()      + QDir::separator() + task.FileName + "." + task.FileExtension;
    } else if (data.exists() && data.mkdir("Videos")) {
        result = data_videos.path() + QDir::separator() + task.FileName + "." + task.FileExtension;
    } else if (root.mkdir("Videos")) {
        result = videos.path()      + QDir::separator() + task.FileName + "." + task.FileExtension;
    }

    return result;
}

bool DownloadManager::ParseMetadata(const QByteArray &raw_data, QString *video_title, QHash<int, QString> *fmt_url_map)
{
    QRegExp title_extractor                      = QRegExp("<meta name=\"title\" content=\"([^\"]+)\">", Qt::CaseInsensitive);
    QRegExp url_encoded_fmt_stream_map_extractor = QRegExp("url_encoded_fmt_stream_map=([^&\"]+)[&\"]",  Qt::CaseInsensitive);

    QString data = QString::fromUtf8(raw_data.data()).simplified();

    if (title_extractor.indexIn(data) != -1) {
        *video_title = ConvertHTMLEntities(title_extractor.cap(1));

        if (url_encoded_fmt_stream_map_extractor.indexIn(data) != -1) {
            QStringList splitted = QUrl::fromPercentEncoding(url_encoded_fmt_stream_map_extractor.cap(1).toUtf8()).split(",", QString::SkipEmptyParts);

            for (int i = 0; i < splitted.size(); i++) {
                QRegExp url_extractor  = QRegExp("url=([^&]+)");
                QRegExp itag_extractor = QRegExp("itag=(\\d+)");

                if (url_extractor.indexIn(splitted.at(i)) != -1 && itag_extractor.indexIn(splitted.at(i)) != -1) {
                    bool ok  = false;
                    int  fmt = itag_extractor.cap(1).toInt(&ok);

                    if (ok) {
                        (*fmt_url_map)[fmt] = QUrl::fromPercentEncoding(url_extractor.cap(1).toUtf8());
                    }
                }
            }

            return true;
        } else {
            return false;
        }
    } else {
        return false;
    }
}

void DownloadManager::RunQueue()
{
    if (MetadataReply == NULL && DownloadReply == NULL) {
        if (NetworkAvailable) {
            for (int i = 0; i < DownloadTasks.size(); i++) {
                if (DownloadTasks.at(i).State != STATE_COMPLETED && DownloadTasks.at(i).State != STATE_PAUSED) {
                    CurrentTask = DownloadTasks.at(i);

                    QUrl url = QUrl::fromEncoded(QString("http://www.youtube.com/watch?v=%1&nomobile=1").arg(CurrentTask.VideoId).toAscii());

                    MetadataReply = NetworkAccessManager->get(QNetworkRequest(url));

                    break;
                }
            }
        } else {
            QTimer::singleShot(QUEUE_RUN_AFTER, this, SLOT(RunQueue()));
        }
    }
}

void DownloadManager::NetworkRequestFinished(QNetworkReply *reply)
{
    if (reply == MetadataReply) {
        if (reply->error() == QNetworkReply::NoError) {
            QString             video_title;
            QHash<int, QString> fmt_url_map;

            if (ParseMetadata(reply->readAll(), &video_title, &fmt_url_map)) {
                if (fmt_url_map.contains(CurrentTask.Fmt)) {
                    bool file_valid;

                    if (ReopenCurrentFile(&file_valid)) {
                        QNetworkRequest request(QUrl::fromEncoded(fmt_url_map[CurrentTask.Fmt].toAscii()));

                        if (file_valid) {
                            request.setRawHeader("Range", QString("bytes=%1-").arg(CurrentTask.Done).toAscii());
                        }

                        DownloadReply = NetworkAccessManager->get(request);

                        connect(DownloadReply, SIGNAL(downloadProgress(qint64, qint64)), SLOT(DownloadProgress(qint64, qint64)));
                        connect(DownloadReply, SIGNAL(readyRead()),                      SLOT(DownloadDataReady()));
                    } else {
                        QTimer::singleShot(QUEUE_RUN_AFTER, this, SLOT(RunQueue()));
                    }
                } else {
                    CurrentTask.State    = STATE_ERROR;
                    CurrentTask.ErrorMsg = "Specified video format is not available";

                    UpdateTask(CurrentTask);

                    QTimer::singleShot(QUEUE_RUN_AFTER, this, SLOT(RunQueue()));
                }
            } else {
                CurrentTask.State    = STATE_ERROR;
                CurrentTask.ErrorMsg = "Specified video format is not available";

                UpdateTask(CurrentTask);

                QTimer::singleShot(QUEUE_RUN_AFTER, this, SLOT(RunQueue()));
            }
        } else {
            if (reply->error() != QNetworkReply::OperationCanceledError) {
                CurrentTask.State    = STATE_ERROR;
                CurrentTask.ErrorMsg = FormatNetworkError(reply->error());

                UpdateTask(CurrentTask);
            }

            QTimer::singleShot(QUEUE_RUN_AFTER, this, SLOT(RunQueue()));
        }

        MetadataReply = NULL;

        reply->deleteLater();
    } else if (reply == DownloadReply) {
        if (reply->error() == QNetworkReply::NoError) {
            QVariant redirect_target = reply->attribute(QNetworkRequest::RedirectionTargetAttribute);

            if (!redirect_target.isNull()) {
                bool file_valid;

                if (ReopenCurrentFile(&file_valid)) {
                    QNetworkRequest request(redirect_target.toUrl());

                    if (file_valid) {
                        request.setRawHeader("Range", QString("bytes=%1-").arg(CurrentTask.Done).toAscii());
                    }

                    DownloadReply = NetworkAccessManager->get(request);

                    connect(DownloadReply, SIGNAL(downloadProgress(qint64, qint64)), SLOT(DownloadProgress(qint64, qint64)));
                    connect(DownloadReply, SIGNAL(readyRead()),                      SLOT(DownloadDataReady()));
                } else {
                    DownloadReply = NULL;

                    QTimer::singleShot(QUEUE_RUN_AFTER, this, SLOT(RunQueue()));
                }
            } else {
                QByteArray data = reply->readAll();

                if (data.isEmpty() || CurrentFile.write(data) != -1) {
                    CurrentTask.Done = CurrentTask.Done + data.size();

                    if (CurrentTask.Size == CurrentTask.Done) {
                        CurrentTask.State    = STATE_COMPLETED;
                    } else {
                        CurrentTask.State    = STATE_ERROR;
                        CurrentTask.Size     = 0;
                        CurrentTask.Done     = 0;
                        CurrentTask.ErrorMsg = "Invalid file size, retrying download";
                    }
                } else {
                    CurrentTask.State    = STATE_ERROR;
                    CurrentTask.ErrorMsg = CurrentFile.errorString();
                }

                UpdateTask(CurrentTask);

                CurrentFile.close();

                DownloadReply = NULL;

                QTimer::singleShot(QUEUE_RUN_AFTER, this, SLOT(RunQueue()));
            }
        } else {
            if (reply->error() != QNetworkReply::OperationCanceledError) {
                CurrentTask.State    = STATE_ERROR;
                CurrentTask.ErrorMsg = FormatNetworkError(reply->error());

                UpdateTask(CurrentTask);
            }

            CurrentFile.close();

            DownloadReply = NULL;

            QTimer::singleShot(QUEUE_RUN_AFTER, this, SLOT(RunQueue()));
        }

        reply->deleteLater();
    }
}

void DownloadManager::DownloadProgress(qint64 received, qint64 total)
{
    if (CurrentTask.Size == 0) {
        CurrentTask.Size = total;
    }
}

void DownloadManager::DownloadDataReady()
{
    QByteArray data = DownloadReply->readAll();

    if (data.isEmpty() || CurrentFile.write(data) != -1) {
        CurrentTask.Done = CurrentTask.Done + data.size();

        UpdateTask(CurrentTask);
    } else {
        CurrentTask.State    = STATE_ERROR;
        CurrentTask.ErrorMsg = CurrentFile.errorString();

        UpdateTask(CurrentTask);

        DownloadReply->abort();
    }
}

bool DownloadManager::ReopenCurrentFile(bool *file_valid)
{
    bool result;

    if (CurrentFile.isOpen()) {
        CurrentFile.close();
    }

    QString file_name = MakeFullFilenameForTask(CurrentTask);

    if (!file_name.isEmpty()) {
        QIODevice::OpenMode mode;

        CurrentFile.setFileName(file_name);

        if (CurrentFile.exists() && CurrentFile.size() == CurrentTask.Done) {
            mode        = QIODevice::ReadWrite | QIODevice::Append;
            *file_valid = true;
        } else {
            mode        = QIODevice::ReadWrite | QIODevice::Truncate;
            *file_valid = false;

            CurrentTask.Size = 0;
            CurrentTask.Done = 0;
        }

        if (CurrentFile.open(mode)) {
            CurrentTask.State = STATE_ACTIVE;

            result = true;
        } else {
            CurrentTask.State    = STATE_ERROR;
            CurrentTask.ErrorMsg = CurrentFile.errorString();

            result = false;
        }
    } else {
        CurrentTask.State    = STATE_ERROR;
        CurrentTask.ErrorMsg = "Video directory not found";

        result = false;
    }

    UpdateTask(CurrentTask);

    return result;
}

void DownloadManager::UpdateTask(const DownloadTask &task)
{
    for (int i = 0; i < DownloadTasks.size(); i++) {
        if (DownloadTasks.at(i).FileName == task.FileName) {
            DownloadTasks.replace(i, task);

            break;
        }
    }
}

bool DownloadManager::LoadTask(const QString &file_name, DownloadTask *task)
{
    QVariant  value;
    QSettings task_file(DownloadsDir.path() + QDir::separator() + file_name, QSettings::IniFormat);

    task->FileName = file_name;
    task->ErrorMsg = "";

    value = task_file.value("State");
    if (!value.isNull()) {
        task->State = value.toInt();

        if (task->State != STATE_COMPLETED && task->State != STATE_PAUSED) {
            task->State = STATE_QUEUED;
        }
    } else {
        return false;
    }

    value = task_file.value("Fmt");
    if (!value.isNull()) {
        task->Fmt = value.toInt();
    } else {
        return false;
    }

    value = task_file.value("Size");
    if (!value.isNull()) {
        task->Size = value.toLongLong();
    } else {
        return false;
    }

    value = task_file.value("Done");
    if (!value.isNull()) {
        task->Done = value.toLongLong();
    } else {
        return false;
    }

    value = task_file.value("VideoId");
    if (!value.isNull()) {
        task->VideoId = value.toString();
    } else {
        return false;
    }

    value = task_file.value("FileExtension");
    if (!value.isNull()) {
        task->FileExtension = value.toString();
    } else {
        return false;
    }

    value = task_file.value("DestinationDisk");
    if (!value.isNull()) {
        task->DestinationDisk = value.toString();
    } else {
        return false;
    }

    value = task_file.value("Title");
    if (!value.isNull()) {
        task->Title = value.toString();
    } else {
        return false;
    }

    return true;
}

void DownloadManager::SaveTask(const DownloadTask &task)
{
    QSettings task_file(DownloadsDir.path() + QDir::separator() + task.FileName, QSettings::IniFormat);

    task_file.setValue("State",           task.State);
    task_file.setValue("Fmt",             task.Fmt);
    task_file.setValue("Size",            task.Size);
    task_file.setValue("Done",            task.Done);
    task_file.setValue("VideoId",         task.VideoId);
    task_file.setValue("FileExtension",   task.FileExtension);
    task_file.setValue("DestinationDisk", task.DestinationDisk);
    task_file.setValue("Title",           task.Title);
}

void DownloadManager::ClearTask(const DownloadTask &task)
{
    QFile task_file(DownloadsDir.path() + QDir::separator() + task.FileName);

    task_file.remove();
}

QString DownloadManager::ConvertHTMLEntities(const QString &string)
{
    QString result(string);

    result = result.replace("&quot;", "\""); result = result.replace("&#34;", "\"");
    result = result.replace("&apos;", "'");  result = result.replace("&#39;", "'");
    result = result.replace("&amp;",  "&");  result = result.replace("&#38;", "&");
    result = result.replace("&lt;",   "<");  result = result.replace("&#60;", "<");
    result = result.replace("&gt;",   ">");  result = result.replace("&#62;", ">");

    return result;
}

QString DownloadManager::FormatNetworkError(QNetworkReply::NetworkError error)
{
    switch (error) {
        case QNetworkReply::NoError                           : return QString("HTTP error: No error");
        case QNetworkReply::ConnectionRefusedError            : return QString("HTTP error: Connection refused");
        case QNetworkReply::RemoteHostClosedError             : return QString("HTTP error: Remote host closed connection");
        case QNetworkReply::HostNotFoundError                 : return QString("HTTP error: Host not found");
        case QNetworkReply::TimeoutError                      : return QString("HTTP error: Connection timed out");
        case QNetworkReply::SslHandshakeFailedError           : return QString("HTTP error: SSL handshake failed");
        case QNetworkReply::ProxyConnectionRefusedError       : return QString("HTTP error: Proxy connection refused");
        case QNetworkReply::ProxyConnectionClosedError        : return QString("HTTP error: Proxy connection closed");
        case QNetworkReply::ProxyNotFoundError                : return QString("HTTP error: Proxy not found");
        case QNetworkReply::ProxyTimeoutError                 : return QString("HTTP error: Proxy timeout");
        case QNetworkReply::ProxyAuthenticationRequiredError  : return QString("HTTP error: Proxy required authentication");
        case QNetworkReply::ContentAccessDenied               : return QString("HTTP error: Access denied");
        case QNetworkReply::ContentOperationNotPermittedError : return QString("HTTP error: Operation not permitted");
        case QNetworkReply::ContentNotFoundError              : return QString("HTTP error: Not found");
        case QNetworkReply::AuthenticationRequiredError       : return QString("HTTP error: Authentication required");
        case QNetworkReply::ContentReSendError                : return QString("HTTP error: Content resend error");
        case QNetworkReply::ProtocolUnknownError              : return QString("HTTP error: Unknown protocol");
        case QNetworkReply::ProtocolInvalidOperationError     : return QString("HTTP error: Invalid protocol operation");
        case QNetworkReply::UnknownNetworkError               : return QString("HTTP error: Unknown network error");
        case QNetworkReply::UnknownProxyError                 : return QString("HTTP error: Unknown proxy error");
        case QNetworkReply::UnknownContentError               : return QString("HTTP error: Unknown content error");
        case QNetworkReply::ProtocolFailure                   : return QString("HTTP error: Protocol failure");
        default                                               : return QString("HTTP error: Unknown error");
    }
}
